﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using GE.Visualisation;
using GE.Physics.Shapes;
using GE.Physics;
using GE.Manager;
using Microsoft.Xna.Framework;

namespace GE.World.Entities
{
    class BulletBillEntity : EnemyEntity
    {
        /// <summary>
        /// Texture's id
        /// </summary>
        int _iIdTexture;

        /// <summary>
        /// Sprite's id
        /// </summary>
        int _iIdSprite;

        /// <summary>
        /// Physic shape
        /// </summary>
        //DynamicShapeRectangle _shape;

        /// <summary>
        /// Direction the entity is facing
        /// </summary>
        int _iDirection;

        Vector2 _v2StartPosition;
        int _iAmplitude;
        float _fFrequency;
        int _iBirthTime;
        int _iHorizontalSpeed;

        /// <summary>
        /// Entity lifetime
        /// </summary>
        int _iLifeTime;

        /// <summary>
        /// Id of the explosion animation
        /// </summary>
        int _iIdAnimationExplosion;

        /// <summary>
        /// The position offset to display the explosion
        /// </summary>
        Vector2 _v2AnimationExplosionOffsetPosition;

        #region Properties Editor

        public int Direction { set { _iDirection = value; } }
        public int Amplitude { set { _iAmplitude = value; } }
        public float Frequency { set { _fFrequency = value; } }
        public int HorizontalSpeed { set { _iHorizontalSpeed = value; } }
        public int LifeTime { set { _iLifeTime = value; } }

#if !GAME
        public static string EDITOR_TILESET { get { return "enemysheet.xml";}}
        public static string EDITOR_SPRITE { get { return "bullet_bills"; } }
#endif

        #endregion

        /// <summary>
        /// Constructor
        /// </summary>
        public BulletBillEntity()
            :base()
        {
            _iIdTexture = -1;
            _iIdSprite = -1;
            _shape = Physics.Physics.Instance.createDynamicRectangle(0, 0, Vector2.Zero, this);
            _shape._bCollisionEnable = false;
            _shape._iGroup = (int)ePhysicGroup.ePhysicEnemy;
            _iIdAnimationExplosion = -1;
            _v2AnimationExplosionOffsetPosition = Vector2.Zero;
        }

        /// <summary>
        /// Initialise the entity. Load all the data which are not loaded during the level loading
        /// </summary>
        public override void init()
        {
            _iIdTexture = Visu.Instance.loadTilset("enemysheet.xml");
            _iIdSprite = Visu.Instance.getSpriteId(_iIdTexture, "bullet_bills");
            _iIdAnimationExplosion = Visu.Instance.getAnimationID("Little_Explosion");

            int iWidth = Visu.Instance.getSpriteWidth(_iIdTexture, _iIdSprite);
            int iHeight = Visu.Instance.getSpriteHeight(_iIdTexture, _iIdSprite);
            _shape.resize(iWidth, iHeight);

            _v2AnimationExplosionOffsetPosition = new Vector2(iWidth / 2, iHeight / 2);
            base.init();
        }
        /// <summary>
        /// Activator
        /// </summary>
        public override void activate()
        {
            _shape._bCollisionEnable = true;
            _shape._v2position = Position;

            _bActive = true;
            _v2StartPosition = Position;

            _iBirthTime = TimeClock.Clock.instance.millisecs;
        }

        public override void update()
        {
            //check if lifetime elapsed
            if (TimeClock.Clock.instance.millisecs >= _iBirthTime + _iLifeTime)
            {
                die();
            }

            float fElapsedTime = (TimeClock.Clock.instance.millisecs - _iBirthTime)/1000f;//elapsed time in seconds
            fElapsedTime = fElapsedTime % MathHelper.TwoPi; //clamp between 0 and 2PI

            float Y = (float)System.Math.Sin(fElapsedTime* _fFrequency) * _iAmplitude;
            float X = _iHorizontalSpeed;
            if(_iDirection < 0)
                X *= -1;

            _v2Position.X += X;
            _v2Position.Y = _v2StartPosition.Y + Y;
            _shape._v2position = _v2Position;

            //check collision
            CollisionResult res = Physics.Physics.Instance.checkFirstRegisteredCollisionEx(_shape, (int)ePhysicGroup.ePhysicPlayer);
            if (res != null)
            {
                res.Entity.hurt(_iDamages);
                die();
            }
        }

        public override void render()
        {
            if (_iDirection < 0)
                Visu.Instance.displaySprite(_iIdTexture, _iIdSprite, ScreenPosition);
            else
                Visu.Instance.displaySprite(_iIdTexture, _iIdSprite, ScreenPosition, Microsoft.Xna.Framework.Graphics.SpriteEffects.FlipHorizontally);
        }

        public override void die()
        {
            _shape._bCollisionEnable = false;
            _bActive = false;
            Manager.ExplosionManager.Instance.activate(_iIdAnimationExplosion, Position + _v2AnimationExplosionOffsetPosition);
        }

        //public override void hurt(int damages)
        //{
        //    die();
        //}
    }
}
